import { TopNav } from "@/components/nav";
import { Button, Card, CardBody, CardHeader, Container } from "@/components/ui";
import Link from "next/link";
import { getSessionOptional } from "@/lib/session";
import { prisma } from "@/lib/prisma";
import { fromMinor } from "@/lib/money";
import { transferToUserId } from "@/lib/ledger";
import { GuestPay } from "@/components/guest/GuestPay";

export default async function PayPosTokenPage({ params }: { params: { token: string } }) {
  const session = await getSessionOptional();

  const charge = await prisma.inStoreCharge.findUnique({ where: { token: params.token } });
  if (!charge) {
    return (
      <>
        <TopNav />
        <Container>
          <div className="mx-auto max-w-xl">
            <Card>
              <CardHeader title="POS payment" subtitle="Not found" />
              <CardBody>
                <div className="text-sm text-zinc-700">This POS payment link is invalid.</div>
              </CardBody>
            </Card>
          </div>
        </Container>
      </>
    );
  }

  const merchant = await prisma.merchantProfile.findUnique({ where: { id: charge.merchantId } });
  if (!merchant) throw new Error("MERCHANT_NOT_FOUND");

  async function pay() {
    "use server";
    if (!session) throw new Error("UNAUTHENTICATED");
    const fresh = await prisma.inStoreCharge.findUnique({ where: { id: charge.id } });
    if (!fresh) throw new Error("NOT_FOUND");
    if (fresh.status === "PAID") return fresh.paidTxId ?? "";

    const tx = await transferToUserId({
      fromUserId: session.uid,
      toUserId: merchant.userId,
      amountMinor: fresh.amount,
      txType: "PAYMENT",
      description: fresh.description ?? `POS payment${fresh.reference ? ` ${fresh.reference}` : ""}`,
      reference: `POS_${fresh.id}`,
    });

    await prisma.inStoreCharge.update({
      where: { id: fresh.id },
      data: { status: "PAID", paidTxId: tx.transaction.id },
    });

    return tx.transaction.id;
  }

  return (
    <>
      <TopNav />
      <Container>
        <div className="mx-auto max-w-xl">
          <Card>
            <CardHeader title="POS payment" subtitle={charge.status === "PAID" ? "Completed" : "Confirm payment"} />
            <CardBody>
              <div className="rounded-xl border border-zinc-200 p-4">
                <div className="text-xs text-zinc-600">Merchant</div>
                <div className="text-sm font-medium">{merchant.tradingName}</div>
                <div className="mt-2 text-xs text-zinc-600">Amount</div>
                <div className="text-2xl font-semibold">P {fromMinor(charge.amount)}</div>
                {charge.reference ? (
                  <div className="mt-2 text-xs text-zinc-600">Reference: {charge.reference}</div>
                ) : null}
                {charge.description ? <div className="mt-1 text-xs text-zinc-600">{charge.description}</div> : null}
                <div className="mt-3 text-xs text-zinc-500">Status: {charge.status}</div>
              </div>

              {charge.status === "PAID" ? (
                <div className="mt-4 text-sm text-green-700">Payment completed.</div>
              ) : (
                <>
                  {session ? (
                    <form action={pay} className="mt-4">
                      <Button type="submit" className="w-full">
                        Pay with LEFA
                      </Button>
                    </form>
                  ) : (
                    <div className="mt-4 grid gap-2">
                      <Link href={`/signin?callbackUrl=/pay/pos/${charge.token}`}>
                        <Button className="w-full">Sign in to pay with LEFA</Button>
                      </Link>
                      <GuestPay mode="POS" token={charge.token} />
                    </div>
                  )}
                </>
              )}

              <div className="mt-4 text-xs text-zinc-500">
                Paying as guest uses OTP and simulates external rails settlement.
              </div>
            </CardBody>
          </Card>
        </div>
      </Container>
    </>
  );
}
